﻿using System;
using System.Collections;
using System.Collections.Generic;
using TMPro;
using UnityEngine;
using UnityEngine.UI;

//WagerRaceInfoMenu handles the wager race info UI
public class WagerRaceInfoMenu : MonoBehaviour
{
    //UI Elements
    public GameObject WagerRaceInfoUI;
    public TMP_Dropdown DifficultiesDropdown;
    public TMP_Text TimeToBeatText;
    public TMP_Text WagerOddsText;
    public IncrementerBox WagerIncrementerBox;
    public TMP_Text ReturnsText;
    public CustomButton AcceptButton;
    public Animator Animator;

    private WagerRaceMission _Mission;

    /// <summary>
    /// Shows the UI for the corresponding wager race mission
    /// </summary>
    /// <param name="mission">The mission to show info for</param>
    public void Show(WagerRaceMission mission)
    {
        _Mission = mission; //Store the mission locally

        WagerIncrementerBox.IncrementAmount = ConfigurationManager.Instance.Missions.WagerRace.WagerIncrement;
        WagerIncrementerBox.CurrentValueChanged -= OnWagerAmountChanged;    //Unsubscribe from old event
        WagerIncrementerBox.CurrentValueChanged += OnWagerAmountChanged;    //Resubscribe to new event

        //Setup the UI
        SetupDropdown();
        OnDifficultyChanged(0);
        WagerIncrementerBox.SetCurrentValue(_Mission.DifficultyData[Constants.MissionDifficulties.Easy].MinWager);
        WagerRaceInfoUI.SetActive(true);

        //Begin the UI SFX audio and animation
        AudioManager.Instance.PlayFile(AudioManager.Instance.LoadedFiles[Constants.UIPageTransToAudioID]);
        Animator.Play("Show");
    }

    /// <summary>
    /// Hides the UI
    /// </summary>
    public void Hide()
    {
        //Play the UI SFX, begin the animation
        AudioManager.Instance.PlayFile(AudioManager.Instance.LoadedFiles[Constants.UIPageTransFromAudioID]);
        Animator.Play("Hide");
    }

    /// <summary>
    /// Handler for when the hide animation completes
    /// </summary>
    public void OnHideAnimationCompleted()
    {
        //All done, hide our UI and the base info menu as well
        WagerRaceInfoUI.SetActive(false);
        GameManager.Instance.MissionInfoMenu.Hide();
    }

    /// <summary>
    /// Handler for when the selected difficulty changes
    /// </summary>
    /// <param name="index">Thhe index of the new difficulty selected</param>
    public void OnDifficultyChanged(int index)
    {
        Constants.MissionDifficulties difficulty = GetSelectedDifficulty();

        if (difficulty != Constants.MissionDifficulties.Sentinel)
        {
            //The selected value is valid, set the UI
            TimeSpan time = TimeSpan.FromMilliseconds(_Mission.DifficultyData[difficulty].Time);
            TimeToBeatText.text = string.Format("{0}:{1:00}", (int)time.TotalMinutes, time.Seconds);

            WagerOddsText.text = "0 : 0";

            WagerIncrementerBox.MinimumAmount = _Mission.DifficultyData[difficulty].MinWager;
            WagerIncrementerBox.MaximumAmount = _Mission.DifficultyData[difficulty].MaxWager;
            WagerIncrementerBox.SetCurrentValue(_Mission.DifficultyData[difficulty].MinWager);
        }
    }

    /// <summary>
    /// Handler for the accept button
    /// </summary>
    public void OnMissionAccepted()
    {
        Constants.MissionDifficulties selectedDifficulty = GetSelectedDifficulty();

        if (selectedDifficulty != Constants.MissionDifficulties.Sentinel)
        {
            //The difficulty is valid, let's start
            MissionsManager.Instance.StartMission(_Mission, selectedDifficulty);
        }

        else
        {
            Debug.LogError("ERROR: Failed to start mission as the selected difficulty was sentinel.");
        }

        Hide(); //Time to hide
    }

    /// <summary>
    /// Handler for the cancel button
    /// </summary
    public void OnMissionCancelled()
    {
        Hide(); //Time to hide
    }

    /// <summary>
    /// Sets up the dropdown by looping through all the enum values and converting them to be displayed
    /// </summary>
    private void SetupDropdown()
    {
        DifficultiesDropdown.ClearOptions();

        foreach (Constants.MissionDifficulties difficulties in Enum.GetValues(typeof(Constants.MissionDifficulties)))
        {
            if (difficulties != Constants.MissionDifficulties.Sentinel)
            {
                DifficultiesDropdown.options.Add(new TMP_Dropdown.OptionData() { text = difficulties.ToString() });
            }
        }

        DifficultiesDropdown.captionText.text = DifficultiesDropdown.options[0].text;
    }

    /// <summary>
    /// Gets the selected difficulty as an enum value by parsing the string
    /// </summary>
    /// <returns>The enum difficulty value if successful, otherwise sentinel</returns>
    private Constants.MissionDifficulties GetSelectedDifficulty()
    {
        Constants.MissionDifficulties retDifficulty = Constants.MissionDifficulties.Sentinel;
        string selectedOptionText = DifficultiesDropdown.options[DifficultiesDropdown.value].text;

        if (!Enum.TryParse(selectedOptionText, out retDifficulty))
        {
            return Constants.MissionDifficulties.Sentinel;
        }

        else
        {
            return retDifficulty;
        }
    }

    private void OnWagerAmountChanged(object sender, IncrementerBox.CurrentValueChangedEventArgs e)
    {
        int wagerAmount = e.NewValue;

        Constants.MissionDifficulties difficulty = GetSelectedDifficulty();

        if (difficulty != Constants.MissionDifficulties.Sentinel)
        {
            if (wagerAmount <= GameManager.Instance.Money)
            {
                if (wagerAmount >= _Mission.DifficultyData[difficulty].MinWager)
                {
                    if (wagerAmount <= _Mission.DifficultyData[difficulty].MaxWager)
                    {
                        //Our difficulty is valid and our amount is within range, compute the new odds

                        int unroundedReturns = wagerAmount + Convert.ToInt32(wagerAmount * _Mission.DifficultyData[difficulty].OddsMultiplier);
                        int returns = Convert.ToInt32(Math.Round(unroundedReturns / (float)ConfigurationManager.Instance.Missions.Core.MoneyRoundFactor) * ConfigurationManager.Instance.Missions.Core.MoneyRoundFactor);

                        int oddsOne = returns - wagerAmount;
                        int oddsTwo = wagerAmount;
                        int gcd = Utilities.GCD(oddsOne, oddsTwo);

                        WagerOddsText.text = (oddsOne / gcd).ToString() + " : " + (oddsTwo / gcd).ToString();
                        _Mission.SetWagerValues(wagerAmount, returns);

                        ReturnsText.text = returns.ToString();
                        AcceptButton.Enable();
                        return;
                    }

                    else
                    {
                        ReturnsText.text = "WAGER ABOVE MAXIMUM!";
                    }
                }

                else
                {
                    ReturnsText.text = "WAGER BELOW MINIMUM!";
                }
            }

            else
            {
                ReturnsText.text = "NOT ENOUGH MONEY!";
            }
        }

        else
        {
            ReturnsText.text = "INVALID DIFFICULTY!";
        }

        ReturnsText.text = "N/A";   //TODO: Think of a better UI solution to display the above then remove this line
        AcceptButton.Disable();
    }

    private void Update()
    {
        if(Input.GetKeyDown(KeyCode.Escape))
        {
            Hide();
        }
    }
}
